package com.kobiton.turbotestAndroid;

import io.appium.java_client.android.AndroidDriver;
import org.openqa.selenium.OutputType;
import org.openqa.selenium.remote.DesiredCapabilities;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.time.LocalDateTime;

public class TurboTest {
    public static void main(String[] args) {

        try {
            Files.createDirectories(Path.of("output"));
        } catch (IOException ignored) {
        }

        try (FileWriter log = new FileWriter("output/test_log.txt", true)) {
            log.write("=== TurboTest Run Started ===\n");
            log.write("Timestamp: " + LocalDateTime.now() + "\n");

            // Required environment variables
            String kobitonUrl = System.getenv("KOBITON_URL");
            String udid = System.getenv("KOBITON_UDID");
            String kobitonSession = System.getenv("KOBITON_SESSION");
            String kobitonUsername = System.getenv("KOBITON_USERNAME");
            String kobitonApiKey = System.getenv("KOBITON_APIKEY");

            // Optional environment variables
            String app = System.getenv("KOBITON_APP");
            String sessionName = System.getenv("KOBITON_SESSION_NAME");
            String sessionDescription = System.getenv("KOBITON_SESSION_DESCRIPTION");
            String kobitonRuntime = System.getenv("KOBITON_RUNTIME");

            // Validate required env vars
            if (kobitonUrl == null || kobitonSession == null || udid == null || kobitonApiKey == null || kobitonUsername == null
                    || kobitonUrl.isEmpty() || kobitonSession.isEmpty() || udid.isEmpty() || kobitonApiKey.isEmpty() || kobitonUsername.isEmpty())  {
                System.err.println("Missing required env vars: KOBITON_UDID, KOBITON_URL, KOBITON_SESSION, KOBITON_USERNAME, or KOBITON_APIKEY");
                return;
            }

            // Hard-code optional env vars if null or empty
            if (app == null || app.isEmpty()) {
                app = "https://kobiton-dev-vn.s3.ap-southeast-1.amazonaws.com/Apps-test/ApiDemos-debug.apk";
            }

            if (sessionName == null || sessionName.isEmpty()) {
                sessionName = "A Turbo Test session for Android";
            }

            if (sessionDescription == null || kobitonUsername.isEmpty()) {
                sessionDescription = "A sample Turbo Test session for Android";
            }

            // Capabilities
            DesiredCapabilities caps = new DesiredCapabilities();
            caps.setCapability("kobiton:sessionName", sessionName);
            caps.setCapability("kobiton:sessionDescription", sessionDescription);
            caps.setCapability("appium:udid", udid);
            caps.setCapability("appium:app", app);
            caps.setCapability("appium:automationName", "UIAutomator2");
            caps.setCapability("kobiton:session", kobitonSession);
            caps.setCapability("appium:username", kobitonUsername);
            caps.setCapability("appium:accessKey", kobitonApiKey);
            caps.setCapability("kobiton:runtime", kobitonRuntime);

            // Initialize driver
            AndroidDriver driver = new AndroidDriver(new URI(kobitonUrl).toURL(), caps);

            // Save page source
            Files.writeString(Path.of("output/page_source.xml"), driver.getPageSource());

            // Screenshot
            File shot = driver.getScreenshotAs(OutputType.FILE);
            Files.copy(shot.toPath(), Path.of("output/screenshot.png"), StandardCopyOption.REPLACE_EXISTING);
            log.write("Screenshot saved\n");

            driver.quit();
            log.write("Completed successfully at " + LocalDateTime.now() + "\n");

            driver.quit();

        } catch (Exception e) {
            try (FileWriter err = new FileWriter("output/error.log", true)) {
                err.write("Exception: " + e.getMessage() + "\n");
                e.printStackTrace(new PrintWriter(err));
            } catch (IOException ignored) {}
            e.printStackTrace();
        }
    }
}




